<?php

class WC_CourierManager_Shipping_Method extends WC_Shipping_Method
{
    const DEFAULT_COUNTRY = "RO";
    const DEFAULT_COUNTY = "B";

    /** @var CourierManagerApi */
    protected $apiClient;

    /**
     * Constructor for your shipping class
     *
     * @access public
     * @param int $instance_id
     */
    public function __construct($instance_id = 0)
    {
        $this->id                 = 'couriermanager_shipping_method'; // Id for your shipping method. Should be uunique.
        $this->method_title       = __( 'Courier Manager' );  // Title shown in admin
        $this->method_description = __( 'Courier software that just works!' ); // Description shown in admin
        $this->enabled            = "yes"; // This can be added as an setting but for this example its forced enabled
        $this->title              = "Courier Manager"; // This can be added as an setting but for this example its forced.
        $this->apiClient = new CourierManagerApi($this->get_option('cm_api_url'), $this->get_option('cm_api_key'));
        $this->init();
        parent::__construct($instance_id);
    }

    /**
     * Init your settings
     *
     * @access public
     * @return void
     */
    function init()
    {
        // Load the settings API
        $this->init_form_fields(); // This is part of the settings API. Override the method to add your own settings
        $this->init_settings(); // This is part of the settings API. Loads settings you previously init.
        // Save settings in admin if you have any defined
        add_action( 'woocommerce_update_options_shipping_' . $this->id, array( $this, 'process_admin_options' ) );
    }

    function init_form_fields()
    {
        $this->form_fields = array(

            'enabled' => array(
                'title' => __('Enabled'),
                'type' => 'checkbox',
                'label' => __('Yes'),
            ),

            'cm_title' => array(
                'title' => __('Title *'),
                'type' => 'text',
                'default' => __('Courier Manager'),
                'required' => true,
            ),

            'cm_api_label' => array(
                'title' => __('Api: '),
                'type' => 'title',
            ),

            'cm_api_url' => array(
                'title' => __('Api URL *'),
                'type' => 'text',
                'default' => __('https://app.curiermanager.ro/cscourier'),
                'required' => true,
            ),

            'cm_api_key' => array(
                'title' => __('Api Key *'),
                'type' => 'text',
                'default' => '',
                'required' => true,

            ),

            'cm_delivery_label' => array(
                'title' => __('Delivery: '),
                'type' => 'title',
            ),

            'cm_shipping_vat' => array(
                'title' => __('Shipping VAT *'),
                'type' => 'number',
                'default' => '19',

            ),

            'cm_delivery_type_code' => array(
                'title' => __('Delivery type code *'),
                'type' => 'text',
                'default' => __('regular'),
                'required' => true,
            ),

            'cm_packaging_type' => array(
                'title' => __('Packaging type'),
                'type' => 'select',
                'label' => '',
                'default' => 'colet',
                'css' => 'width:350px;',
                'options' => array(
                    'plic' => __('Envelope'),
                    'colet' => __('Package')
                ),

            ),

            'cm_payment_awb' => array(
                'title' => __('Payment AWB at destination'),
                'type' => 'select',
                'label' => '',
                'default' => '0',
                'css' => 'width:350px;',
                'options' => array(
                    '0' => __('No'),
                    '1' => __('Yes')
                ),
            ),

            'cm_cod_type' => array(
                'title' => __('COD type'),
                'type' => 'select',
                'label' => '',
                'default' => 'cont',
                'css' => 'width:350px;',
                'options' => array(
                    'cont' => __('Bank account'),
                    'cash' => __('Cash')
                ),
            ),

            'cm_goods_insured' => array(
                'title' => __('Goods Insured'),
                'type' => 'select',
                'label' => '',
                'default' => '0',
                'css' => 'width:350px;',
                'options' => array(
                    '0' => __('No'),
                    '1' => __('Yes')
                ),
            ),

            'cm_delivery_price_label' => array(
                'title' => __('Amount rules: '),
                'type' => 'title',
            ),

            'cm_minim_amount' => array(
                'title' => __('Minimum amount of free shipping'),
                'type' => 'price'
            ),

            'cm_fix_amount' => array(
                'title' => __('Fix amount of shipping'),
                'type' => 'price'
            )
        );
    }

    /**
     * @param array $package
     * @return array
     */
    private function getDeliveryData($package = array())
    {
        $countryStates = WC()->countries->get_shipping_country_states();
        $countries = WC()->countries->get_shipping_countries();
        $countryId = $package['destination']['country'];
        $stateId = $package['destination']['state'];

        $countryName = self::DEFAULT_COUNTRY;
        if (!empty($countries[$countryId])) {
            $countryName = $countries[$countryId];
        }

        $regionName = self::DEFAULT_COUNTY;
        if (!empty($countryStates[$countryId][$stateId])) {
            $regionName = $countryStates[$countryId][$stateId];
        }

        // Address
        $address = $package['destination']['address'];
        if (!empty($package['destination']['address_1'])) {
            $address.= ", " . $package['destination']['address_1'];
        }

        if (!empty($package['destination']['address_2'])) {
            $address.= ", " . $package['destination']['address_2'];
        }

        $cartPrice = wc_prices_include_tax() ? WC()->cart->get_cart_contents_total() + WC()->cart->get_cart_contents_tax() : WC()->cart->get_cart_contents_total();

        // API call data
        return array(
            "use_default_from_address" => "true",
            "type" => $this->get_option('cm_packaging_type'),
            "service_type" => $this->get_option('cm_delivery_type_code'),
            "cnt" => 1,//WC()->cart->get_cart_contents_count(),
            "weight" => WC()->cart->get_cart_contents_weight(),
            "to_country" => $countryName,
            "to_county" => $regionName,
            "to_city" => $package['destination']['city'],
            "to_address" => $address,
            "to_zipcode" => $package['destination']['postcode'],
            "ramburs" => $cartPrice,
            "ramburs_type" => $this->get_option('cm_cod_type'),
        );
    }

    /**
     * @param array $package
     * @return int|mixed|null
     */
    private function getShippingCost($package = array())
    {
        // Check if total is upper minim free sum
        if ($this->get_option('cm_minim_amount') && WC()->cart->get_subtotal() >= $this->get_option('cm_minim_amount')) {
            return 0;
        }

        // Return fix amount if is set
        if ($this->get_option('cm_fix_amount') && $this->get_option('cm_fix_amount') > 0) {
            return $this->get_option('cm_fix_amount');
        }

        // Call API
        $deliveryData = $this->getDeliveryData($package);
        $apiPrice = $this->apiClient->priceAwb($deliveryData);

        // Check if response is ok
        if (!isset($apiPrice->status) || $apiPrice->status != "done") {
            return null;
        }

        $taxes = $apiPrice->data->price * (  $this->get_option('cm_shipping_vat') / 100 );

        // Return delivery price
        return $apiPrice->data->price + $taxes;
    }


    /**
     * calculate_shipping function.
     *
     * @access public
     * @param mixed $package
     * @return void
     */
    public function calculate_shipping( $package = array() )
    {
        $rate = array(
            'id' => $this->id,
            'label' => $this->get_option('cm_title'),
            'cost' => $this->getShippingCost($package),
            'calc_tax' => 'per_order'
        );

        // Register the rate
        $this->add_rate( $rate );
    }
}